<?php
// File: local/learningpath/learner_path_view.php
require_once('../../config.php');
require_login();

global $DB, $USER, $PAGE, $OUTPUT, $CFG;

$pathid = required_param('id', PARAM_INT);

// learner-only
function is_user_learner_role($userid = null) {
    if (!$userid) global $USER;
    $roles = get_user_roles(context_system::instance(), $userid ? $userid : $USER->id, true);
    foreach ($roles as $r) {
        if (in_array($r->shortname, ['student','learner','participant','studentmember'])) return true;
    }
    return false;
}

if (!is_user_learner_role($USER->id)) {
    $PAGE->set_context(context_system::instance());
    echo $OUTPUT->header();
    echo $OUTPUT->notification('Access denied — learners only.', 'notifyproblem');
    echo $OUTPUT->footer();
    exit;
}

$PAGE->set_url(new moodle_url('/local/learningpath/learner_path_view.php', ['id' => $pathid]));
$PAGE->set_context(context_system::instance());

// include CSS
$cssfile = new moodle_url('/local/learningpath/learner_styles.css');
$PAGE->requires->css($cssfile);

// fetch learning path
$lp = $DB->get_record('learning_paths', ['id' => $pathid], '*', MUST_EXIST);

// fetch sections ordered by position
$sections = $DB->get_records('learning_path_sections', ['learningpathid' => $pathid], 'position ASC');

$PAGE->set_title(format_string($lp->name));
$PAGE->set_heading(format_string($lp->name));

echo $OUTPUT->header();

echo '<div class="lp-container">';

// header / hero
echo '<div class="lp-hero">';
echo '<div>';
echo '<div class="title">' . format_string($lp->name) . '</div>';
echo '<div class="sub">' . ($lp->description ? format_text($lp->description, FORMAT_HTML) : '') . '</div>';
echo '</div>';

// find first unlocked course to link Start Learning
// sequencing rule: first section unlocked by default; subsequent unlocked only if all courses in previous section completed
$first_unlocked_course_url = null;
$found = false;
$prev_section_all_completed = true;
foreach ($sections as $sindex => $s) {
    // fetch courses in this section ordered by sortorder
    $courses = $DB->get_records('learning_path_courses', ['sectionid' => $s->id], 'sortorder ASC');
    // check if this section is unlocked: if sindex==0 unlocked, else unlocked only if prev_section_all_completed is true
    $section_unlocked = ($sindex == 0) ? true : $prev_section_all_completed;

    // evaluate courses completion within this section to update $prev_section_all_completed
    $all_completed_here = true;
    foreach ($courses as $lc) {
        $cc = $DB->get_record('course_completions', ['userid' => $USER->id, 'course' => $lc->courseid]);
        if (!$cc || empty($cc->timecompleted)) {
            $all_completed_here = false;
            // if this section is unlocked and we haven't found first unlocked course yet, set it to this course
            if ($section_unlocked && !$found) {
                $first_unlocked_course_url = new moodle_url('/course/view.php', ['id' => $lc->courseid]);
                $found = true;
            }
        } else {
            // completed - continue
            if (!$found && $section_unlocked) {
                // skip as it's completed; but if no unlocked found yet, continue searching
            }
        }
    }
    // if section has no courses, consider it completed (so next can unlock)
    if (empty($courses)) {
        $all_completed_here = true;
    }
    $prev_section_all_completed = $all_completed_here;
    if ($found) break;
}

// fallback: link to view page if nothing unlocked
if (!$first_unlocked_course_url) {
    $first_unlocked_course_url = new moodle_url('/local/learningpath/learner_path_view.php', ['id' => $pathid]);
}

echo '<div style="text-align:right">';
echo '<a class="start-cta" href="' . $first_unlocked_course_url->out(false) . '">Start Learning</a>';
echo '</div>';

echo '</div>'; // hero

// Tabs
echo '<div class="lp-tabs" id="lpTabs">';
$idx = 0;
foreach ($sections as $s) {
    $name = $s->sectionname ? $s->sectionname : 'Section ' . ($idx + 1);
    echo '<div class="lp-tab ' . ($idx===0 ? 'active' : '') . '" data-id="' . $s->id . '">' . format_string($name) . '</div>';
    $idx++;
}
echo '</div>';

// Section contents
$idx = 0;
foreach ($sections as $s) {
    // fetch courses in this section ordered
    $courses = $DB->get_records('learning_path_courses', ['sectionid' => $s->id], 'sortorder ASC');

    // determine if this section is unlocked:
    // first section unlocked; others unlocked only if all courses in previous section(s) are completed
    $section_unlocked = true;
    if ($idx > 0) {
        // check previous section index
        $previdx = $idx - 1;
        // get prev section
        $prevsection = array_values($sections)[$previdx];
        // check all courses in prevsection completed
        $prevcourses = $DB->get_records('learning_path_courses', ['sectionid' => $prevsection->id]);
        $prev_all_done = true;
        foreach ($prevcourses as $pc) {
            $cc = $DB->get_record('course_completions', ['userid' => $USER->id, 'course' => $pc->courseid]);
            if (!$cc || empty($cc->timecompleted)) { $prev_all_done = false; break; }
        }
        $section_unlocked = $prev_all_done;
    }

    $display_style = ($idx === 0) ? '' : 'style="display:none"';
    echo '<div id="section_content_' . $s->id . '" class="section-wrap" ' . $display_style . '>';

    // compute section-level progress (#courses completed / total)
    $total = count($courses);
    $completed = 0;
    foreach ($courses as $c) {
        $cc = $DB->get_record('course_completions', ['userid' => $USER->id, 'course' => $c->courseid]);
        if ($cc && !empty($cc->timecompleted)) $completed++;
    }
    $secpct = ($total > 0) ? intval(($completed / $total) * 100) : 100;

    echo '<div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:12px">';
    echo '<div><strong>' . format_string($s->sectionname) . '</strong></div>';
    echo '<div style="display:flex;align-items:center;gap:10px">';
    echo '<div class="small-badge">Section progress: ' . $secpct . '%</div>';
    if (!$section_unlocked) {
        echo '<div class="lock-badge">Locked</div>';
    } else {
        // if unlocked and all completed show completed badge
        if ($total > 0 && $completed === $total) {
            echo '<div class="completed-badge">Completed</div>';
        }
    }
    echo '</div></div>';

    // show progress bar
    echo '<div class="lp-progress-wrap" style="margin-bottom:12px">';
    echo '<div class="lp-progress" style="height:8px"><i style="width:' . $secpct . '%;"></i></div>';
    echo '</div>';

    // courses list
    if (empty($courses)) {
        echo '<div class="card p-3">No courses in this section.</div>';
    } else {
        foreach ($courses as $c) {
            // determine course status
            $cc = $DB->get_record('course_completions', ['userid' => $USER->id, 'course' => $c->courseid]);
            $iscompleted = ($cc && !empty($cc->timecompleted));
            $courseurl = new moodle_url('/course/view.php', ['id' => $c->courseid]);

            // if section locked, course button is disabled
            $btnclass = 'course-start';
            $btndisabled = false;
            $statusbadge = '';

            if (!$section_unlocked) {
                $btndisabled = true;
                $btnclass .= ' disabled';
                $statusbadge = '<div class="lock-badge">Locked</div>';
            } else {
                if ($iscompleted) {
                    $statusbadge = '<div class="completed-badge">Completed</div>';
                } else {
                    // enabled
                    $statusbadge = '<div class="small-badge">Not started</div>';
                }
            }

            // simple initials for thumb
            $initials = htmlspecialchars(substr($DB->get_field('course', 'fullname', ['id' => $c->courseid]), 0, 2));

            echo '<div class="course-row">';
            echo '  <div class="course-left">';
            echo '    <div class="course-thumb">' . $initials . '</div>';
            echo '    <div class="course-meta">';
            echo '      <h4>' . format_string($DB->get_field('course', 'fullname', ['id' => $c->courseid])) . '</h4>';
            // course summary if available
            $summary = $DB->get_field('course', 'summary', ['id' => $c->courseid]);
            echo '      <p>' . ($summary ? strip_tags(format_text($summary, FORMAT_HTML)) : '&nbsp;') . '</p>';
            echo '    </div>';
            echo '  </div>';

            echo '  <div style="display:flex;align-items:center;gap:12px">';
            echo $statusbadge;
            if ($btndisabled) {
                echo '<a class="' . $btnclass . ' disabled" href="javascript:void(0);">Locked</a>';
            } else {
                if ($iscompleted) {
                    echo '<a class="course-start" href="' . $courseurl->out(false) . '">View</a>';
                } else {
                    echo '<a class="course-start" href="' . $courseurl->out(false) . '">Start</a>';
                }
            }
            echo '  </div>';
            echo '</div>';
        }
    }

    echo '</div>'; // section wrap

    $idx++;
}

echo '</div>'; // container

// JS to switch tabs
$js = <<<JS
(function(){
  var tabs = document.querySelectorAll('.lp-tab');
  if (!tabs) return;
  tabs.forEach(function(tab){
    tab.addEventListener('click', function(){
      var id = this.getAttribute('data-id');
      // remove active
      tabs.forEach(t=>t.classList.remove('active'));
      this.classList.add('active');
      // hide contents
      document.querySelectorAll('[id^="section_content_"]').forEach(el => el.style.display = 'none');
      var show = document.getElementById('section_content_' + id);
      if (show) show.style.display = '';
    });
  });
})();
JS;

$PAGE->requires->js_init_code($js);

echo $OUTPUT->footer();
exit;
